/*
================================================================================
    makePhotoFrame.jsx
    v.1.0.0
    2024.05.21
    nuts2u
================================================================================
*/

//変数
var doc = null;
var imgLyr = null;
var bkLyr = null;
var whLyr = null;
var capLyr = null;;

//初期の定規単位とテキストサイズ単位を記憶しておく
const origRulerUnit = app.preferences.rulerUnits;
const origTypeUnit  = app.preferences.typeUnits;

try {
    //定規単位をピクセルに、テキストサイズ単位をポイントに変更
    app.preferences.rulerUnits = Units.PIXELS;
    app.preferences.typeUnits  = TypeUnits.PIXELS;

    //画像ファイル選択ダイアログ、キャンセルされるとnullが返ってくる
    const jpgFiles = selectFilesDialog("*.jpg");

    if (jpgFiles == null) {
        alert("No file is selected.");
    } else {
        //ユーザー設定オブジェクトを生成
        const defaultUserSetting = {
            //キャプションテキストの内容
            caption: "© 2024 nuts2u. NIKON Z f, CARL ZEISS Makro-Planar T* 2/50 ZF.2.",
            //フォント、app.font.postScriptNameで指定
            //font: "MendlSans-DuskThin",
            font: "YuGothic-Light",
            //フォントサイズ
            fontSize: 1.0,
            //キャプションテキストと写真との隙間
            captionOffset: 1.0,
            //白フチのサイズ(写真長手基準)
            whFrameThickness: 2.5,
            //白フチのアスペクト比：W/H (1/1 = square or 4/3, 3/2, 16/9, etc...) 
            aspectRatioW: 1.0,
            aspectRatioH: 1.0,
            aspectRatio: 1.0,
            //リサイズ後の長手側(px)、<=0：リサイズしない
            resize: 2000,
            //白フチの色(背景色)、カラーコードで指定、3桁でもok、#なくてもok
            whColorCode: "#ffffff",
            //黒フチの色(境界色)、カラーコードで指定、3桁でもok、#なくてもok
            bkColorCode: "#000000",
            //黒フチの太さ(写真長辺に対する％)、0：黒フチを付けない
            bkLineWidth: .75,
        };

        //ユーザー設定を入力するGUIを起動、キャンセルされるとnullが返ってくる
        //Cancel=nullの場合、エラーをthrowして終了処理
        const userSetting = setupDialog(defaultUserSetting);
        if (userSetting == null){
            throw new Error("Canceled.");  
        }

        //開いているドキュメントがあればすべて閉じておく
        while (app.documents.length) {
            app.activeDocument.close()
        }

        for (var fileNum=0; fileNum<jpgFiles.length; fileNum++) {
            //ファイルを開く
            app.open(jpgFiles[fileNum]);
            doc = app.activeDocument;
            //写真を背景からレイヤに変更
            imgLyr = makePhotoLayer(doc, userSetting);
            //黒フチを追加
            bkLyr = makeBlackFrameLayer(doc, userSetting);
            //白フチを追加
            whLyr = makeWhiteFrameLayer(doc, userSetting);
            //キャプションを追加
            if (bkLyr != null){
                capLyr = makeCaptionLayer(doc, userSetting, bkLyr);
            } else {
                capLyr = makeCaptionLayer(doc, userSetting, imgLyr);
            }
            //リサイズ
            resizeImage(doc, userSetting);
            //画像保存
            savePicture(doc, userSetting);
            //ファイルを閉じる
            app.activeDocument.close(SaveOptions.DONOTSAVECHANGES);
            //リセット(どうやらPhotoshopではfor文でconstが使えない)
            doc = null;
            imgLyr = null;
            bkLyr = null;
            whLyr = null;
            capLyr = null;
        }
    }
    alert("Complete.");
} catch(e) {
    alert(e);
} finally {
     //定規単位を戻しておく
    app.preferences.rulerUnits = origRulerUnit;
    app.preferences.typeUnits  = origTypeUnit;
}

function selectFilesDialog(extensionStr){
    const selectedFolder = Folder.selectDialog("Select a folder.");
    if (selectedFolder == null) {
        return null;
    } else {
        const files = selectedFolder.getFiles(extensionStr);
        //alert(files.length+" file(s) found.");
        return files;
    }
}

function setupDialog(userSettingObj) {
    //*****GUIウィンドウ*****
    const guiWindow = new Window("dialog", "User Setup", {width: 500, height: 500, x: 500, y: 300});
    
    //*****パネル1*****
    const panel1 = guiWindow.add("panel", {width: 470, height: 160, x: 15, y: 15}, "Caption Setting");

      //1-1.キャプションテキスト内容
      const captionETxt = panel1.add("edittext", {width: 440, height: 10, x: 15, y: 20}, userSettingObj.caption);
      captionETxt.active = true;
    
      //1-2.フォント
      var fontnameList = [""];
      for (fontNum=0; fontNum<app.fonts.length; fontNum++){
          fontnameList.push(app.fonts[fontNum].name);
      }
      panel1.add("statictext", {width: 40, height: 20, x: 15, y: 55}, "Font");
      const fontnameDDList = panel1.add("dropdownlist", {width: 220, height: 20, x: 235, y: 55}, fontnameList);
      fontnameDDList.selection = 0;
    
      //1-3.フォントサイズ
      panel1.add("statictext", {width: 350, height: 20, x: 15, y: 85}, "Font size");
      const fontSizeETxt = panel1.add("edittext", {width: 60, height: 10, x: 395, y: 85}, userSettingObj.fontSize);
    
      //1-4.キャプションの写真に対する位置オフセット量
      panel1.add("statictext", {width: 350, height: 20, x: 15, y: 115}, "Alignment offset of the caption");
      const captionOffsetETxt = panel1.add("edittext", {width: 60, height: 10, x: 395, y: 115}, userSettingObj.captionOffset);
    
    //*****パネル2*****
    const panel2 = guiWindow.add("panel", {width: 470, height: 220, x: 15, y: 190}, "Frame Setting");
    
      //2-1.写真長手に対する白フチのサイズ
      panel2.add("statictext", {width: 270, height: 20, x: 15, y: 20}, "The minimum thickness of white frame relative to image");
      const frameScaleETxt = panel2.add("edittext", {width: 60, height: 20, x: 395, y: 20}, userSettingObj.whFrameThickness);
      //panel2.add("statictext", {width: 150, height: 10, x: 320, y: 25}, "");
    
      //2-2.白フチのアスペクト比
      panel2.add("statictext", {width: 350, height: 20, x: 15, y: 50}, "Frame aspect ratio, W / H (W=0: fixed width, H=0: bottom margin)");
      const aspRatioWETxt = panel2.add("edittext", {width: 10, height: 20, x: 365, y: 50}, userSettingObj.aspectRatioW);
      panel2.add("statictext", {width: 10, height: 20, x: 405, y: 50}, "/");
      const aspRatioHETxt = panel2.add("edittext", {width: 10, height: 20, x: 425, y: 50}, userSettingObj.aspectRatioH);
    
      //2-3.リサイズ
      panel2.add("statictext", {width: 350, height: 20, x: 15, y: 80}, "Resize length (0: no resizing) [px]");
      const resizeETxt = panel2.add("edittext", {width: 60, height: 20, x: 395, y: 80}, userSettingObj.resize);
    
      //2-4.白フチ(背景色)
      panel2.add("statictext", {width: 350, height: 20, x: 15, y: 110}, "White background color code");
      const whColorCodeETxt = panel2.add("edittext", {width: 60, height: 20, x: 395, y: 110}, userSettingObj.whColorCode);
    
      //2-5.黒フチ(境界色)
      panel2.add("statictext", {width: 350, height: 20, x: 15, y: 140}, "Black border color code");
      const bkColorCodeETxt = panel2.add("edittext", {width: 60, height: 20, x: 395, y: 140}, userSettingObj.bkColorCode);
    
      //2-6.境界線の太さ
      panel2.add("statictext", {width: 350, height: 20, x: 15, y: 170}, "Black border width (0: no border)");
      const bkWidthETxt = panel2.add("edittext", {width: 60, height: 20, x: 395, y: 170}, userSettingObj.bkLineWidth);
    
    //cancelボタンとokボタン
    const okBtn     = guiWindow.add("button", {width: 100, height: 25, x: 125, y: 450}, "OK"    , {name:"ok"    });
    const cancelBtn = guiWindow.add("button", {width: 100, height: 25, x: 275, y: 450}, "Cancel", {name:"cancel"});
    
    //*****動作*****
    //cancelボタンクリック時の動作
    cancelBtn.onClick = function () {
        //userSettingをnullに
        userSettingObj = null;
        guiWindow.close();
    };
    
    //okボタンクリック時の動作
    okBtn.onClick = function () {
        //userSettingに入力値を設定
        userSettingObj.caption = captionETxt.text;
        if (fontnameDDList.selection.index > 0) {
            //今後制御に使うのはapp.font.nameではなくapp.font.postScriptName
            //ドロップダウンリスト先頭に空行を入れている分-1で戻す
            userSettingObj.font = app.fonts[fontnameDDList.selection.index-1].postScriptName;
        } else {
            //do nothing デフォルト値を使う
        }
        userSettingObj.fontSize         = parseFloat(fontSizeETxt.text);
        userSettingObj.captionOffset    = parseFloat(captionOffsetETxt.text);
        userSettingObj.whFrameThickness = parseFloat(frameScaleETxt.text);
        userSettingObj.aspectRatioW     = parseFloat(aspRatioWETxt.text);
        userSettingObj.aspectRatioH     = parseFloat(aspRatioHETxt.text);
        if(userSettingObj.aspectRatioH == 0) {
            userSettingObj.aspectRatio = null;
        } else {
            userSettingObj.aspectRatio = userSettingObj.aspectRatioW/userSettingObj.aspectRatioH;
        }
        userSettingObj.resize      = parseInt(resizeETxt.text);
        userSettingObj.whColorCode = whColorCodeETxt.text;
        userSettingObj.bkColorCode = bkColorCodeETxt.text;
        userSettingObj.bkLineWidth = parseFloat(bkWidthETxt.text);
        /*****debug*****
        alert("caption          : " + userSettingObj.caption          + "\n" + 
              "font             : " + userSettingObj.font             + "\n" + 
              "fontSize         : " + userSettingObj.fontSize         + "\n" + 
              "captionOffset    : " + userSettingObj.captionOffset    + "\n" + 
              "whFrameThickness : " + userSettingObj.whFrameThickness + "\n" + 
              "aspectRatio      : " + userSettingObj.aspectRatio      + "\n" + 
              "resize           : " + userSettingObj.resize           + "\n" + 
              "whColorCode      : " + userSettingObj.whColorCode      + "\n" +
              "bkColorCode      : " + userSettingObj.bkColorCode      + "\n" + 
              "bkLineWidth      : " + userSettingObj.bkLineWidth);
         ***************/
        guiWindow.close();
    };

    //表示
    guiWindow.center();
    guiWindow.show();
    //出力
    return userSettingObj;
}

function makePhotoLayer(docObj){
    const lyrs = docObj.artLayers;
    const imgLyr = lyrs[lyrs.length-1];
    if (imgLyr.isBackgroundLayer) {
        //不透明度を設定するとレイヤになる
        imgLyr.opacity = 100;
    }
    imgLyr.name = "Photograph layer";
    return imgLyr;
}

function makeBlackFrameLayer(docObj, userSettingObj){
    //変数
    var imgW = .0;
    var imgH = .0;
    var bkLyr = null;
    //colorcodeをSolidColorに変換
    const bkColor = convertColorcodeToSolidColor(userSettingObj.bkColorCode);
    //元写真のサイズ
    const origImgW = docObj.width;
    const origImgH = docObj.height;
    imgW = origImgW;
    imgH = origImgH;
    //縦写真は90度左回転（最後に戻す）
    if (origImgH > origImgW) {
        docObj.rotateCanvas(-90);
        imgW = origImgH;
        imgH = origImgW;
    }
    //黒フチの長さ
    const bkLineWidth = imgW * (userSettingObj.bkLineWidth/100);
    const bkFrameW = imgW + bkLineWidth;
    const bkFrameH = imgH + bkLineWidth;
    //黒フチの方が大きいとき、背景レイヤを作成
    if (bkFrameW > imgW){
        bkLyr = createFillBackgroundLayer(docObj, bkFrameW, bkFrameH, bkColor, "Border Color layer");
    }
    //縦写真は元に戻す
    if (origImgH > origImgW) {
        docObj.rotateCanvas(90);
    }
    return bkLyr;
    //黒フチを作っていない時はnullが返る
}

function convertColorcodeToSolidColor(colorcodeStr) {
    //colorcodeをRGBの値に変換
    const rgbValue = convertColorcodeToRGB(colorcodeStr);
    //SolidColorオブジェクトを作って、RGBで色を指定
    const solidColorObj = new SolidColor();
      solidColorObj.rgb.red   = rgbValue.r;
      solidColorObj.rgb.green = rgbValue.g;
      solidColorObj.rgb.blue  = rgbValue.b;
    return solidColorObj;
}

function convertColorcodeToRGB(colorcodeStr) {
    if (colorcodeStr.split('')[0] == '#') {
        colorcodeStr = colorcodeStr.substring(1);
    }
    if (colorcodeStr.length == 6) {
        //do nothing
    } else if (colorcodeStr.length == 3) {
        const codeArr = colorcodeStr.split('');
        colorcodeStr = codeArr[0] + codeArr[0] + codeArr[1] + codeArr[1] + codeArr[2] + codeArr[2];
    } else {
        colorcodeStr = "000000";
    }
    //var r = parseInt(colorcodeStr.substring(0, 2), 16);
    //var g = parseInt(colorcodeStr.substring(2, 4), 16);
    //var b = parseInt(colorcodeStr.substring(4, 6), 16);
    //return [r, g, b];
    const rgb = {
        r: parseInt(colorcodeStr.substring(0, 2), 16),
        g: parseInt(colorcodeStr.substring(2, 4), 16),
        b: parseInt(colorcodeStr.substring(4, 6), 16)
    }
    return rgb;
}

function createFillBackgroundLayer(docObj, bgLyrW, bgLyrH, fillColor, lyrNameStr) {
    //カンバスサイズを変更
    const newCanvasW = Math.max(docObj.width , bgLyrW);
    const newCanvasH = Math.max(docObj.height, bgLyrH);
    docObj.resizeCanvas(newCanvasW, newCanvasH);
    //レイヤを追加
    const lyrs = docObj.artLayers;
    const bgLyr  = lyrs.add();
    bgLyr.name = lyrNameStr;
    //全選択、塗りつぶし、選択解除
    docObj.selection.selectAll();
    docObj.selection.fill(fillColor);
    docObj.selection.deselect();
    //塗り潰したレイヤを一番下に移動
    bgLyr.move(lyrs[lyrs.length-1], ElementPlacement.PLACEAFTER);
    return bgLyr;
}

function makeWhiteFrameLayer(docObj, userSettingObj) {
    //変数
    var whFrameW = .0;
    var whFrameH = .0;
    var whFrameThickness = .0;
    var whLyr = null;
    //colorcodeをSoliColorに変換
    const whColor = convertColorcodeToSolidColor(userSettingObj.whColorCode);
    //元写真のサイズ
    const imgW = docObj.width;
    const imgH = docObj.height;
    const whAspectRatio = userSettingObj.aspectRatio;
    //Polaroid風以外
    if (whAspectRatio != null) {
        //白フチが均等幅の場合
        if (whAspectRatio == 0) {
            whFrameThickness = Math.max(imgW, imgH) * (userSettingObj.whFrameThickness/100);
            whFrameW = imgW + 2*whFrameThickness;
            whFrameH = imgH + 2*whFrameThickness;
        //白フチにアスペクト比が設定されていて
        //縦長の場合
        } else if (whAspectRatio < 1) {
            whFrameW = Math.max(imgW, imgH) * (1 + 2*userSettingObj.whFrameThickness/100);
            whFrameH = whFrameW / whAspectRatio;
        //横長の場合
        } else if (whAspectRatio >= 1) {
            whFrameH = Math.max(imgW, imgH) * (1 + 2*userSettingObj.whFrameThickness/100);;
            whFrameW = whFrameH * whAspectRatio;
        }
        //レイヤの作成
        if (whFrameW>imgW && whFrameH>imgH){
            whLyr = createFillBackgroundLayer(docObj, whFrameW, whFrameH, whColor, "Background Color layer");
        }
    //Polaroid風の処理
    } else {
        whFrameThickness = (Math.max(imgW, imgH)) * (userSettingObj.whFrameThickness/100);
        whLyr = createPolaroidBackgroundLayer(docObj, whFrameThickness, whColor, "Background Color layer");
    }
    return whLyr;
    //白フチを作っていない時はnullが返る
}

function createPolaroidBackgroundLayer(docObj, bgFrameThickness, fillColor, lyrNameStr){
    //デフォルト値
    const imgOffsetCoef = 5.0;
    //サイズの計算
    //写真下側は bgFrameThickness x 6 の余白を付ける
    const bgLyrW = docObj.width  + 2 * bgFrameThickness;
    const bgLyrH = docObj.height + (2 + imgOffsetCoef) * bgFrameThickness;
    //カンバスサイズを変更
    const newCanvasW = Math.max(docObj.width , bgLyrW);
    const newCanvasH = Math.max(docObj.height, bgLyrH);
    docObj.resizeCanvas(newCanvasW, newCanvasH);
    //レイヤを追加
    const lyrs = docObj.artLayers;
    const bgLyr  = lyrs.add();
    bgLyr.name = lyrNameStr;
    //全選択、塗りつぶし、選択解除
    docObj.selection.selectAll();
    docObj.selection.fill(fillColor);
    docObj.selection.deselect();
    //塗り潰したレイヤを一番下に移動
    bgLyr.move(lyrs[lyrs.length-1], ElementPlacement.PLACEAFTER);
    if (bgFrameThickness > 0){
        for (lyrNum=0; lyrNum<lyrs.length-1; lyrNum++) { 
            lyrs[lyrNum].translate(0, -1/2*imgOffsetCoef*bgFrameThickness);
        }
    }
    return bgLyr;
}

function makeCaptionLayer(docObj, userSettingObj, refLyr){
    //デフォルト値
    const fontSizeCoef = .018; //(110/6000)
    const capOffsetCoef = .003; //(0.3%)
    //参照レイヤのサイズを調べる
    //boundsは境界座標
    //0が左端（x1）、1が上端（y1）、2が右端（x2）、3が下端（y2）
    //const imgLyr = lyrs.getByName("Photograph layer");
    const origImgW = refLyr.bounds[2] - refLyr.bounds[0];
    const origImgH = refLyr.bounds[3] - refLyr.bounds[1];
    const origImgLength = Math.max(origImgW, origImgH);
    //縦写真は90度左回転（最後に戻す）
    if (origImgH>origImgW && userSettingObj.aspectRatio!=null) {
        docObj.rotateCanvas(-90);
    }
    //テキストレイヤを作成
    const txtLyr = docObj.artLayers.add();
    txtLyr.name = "Caption layer";
    txtLyr.kind = LayerKind.TEXT; 
    //アンチエイリアス: 強く
    txtLyr.textItem.antiAliasMethod = AntiAlias.STRONG;
    txtLyr.textItem.font = userSettingObj.font;
    //px-->ptに単位変換、pt = px * 72/dpi
    const dpi = docObj.resolution;
    txtLyr.textItem.size = ((parseFloat(origImgLength)) * fontSizeCoef) * userSettingObj.fontSize * 72/dpi;
    txtLyr.textItem.contents = userSettingObj.caption;
    //キャプションを参照レイヤ（写真レイヤ）右下に移動
    //参照レイヤの右下の座標
    const imgX2 = refLyr.bounds[2];
    const imgY2 = refLyr.bounds[3]; 
    //テキストレイヤの右上の座標
    const txtX2 = txtLyr.bounds[2];
    const txtY1 = txtLyr.bounds[1];
    //まずは上記２つの座標が一致するようにテキストレイヤを移動
    txtLyr.translate(imgX2 - txtX2, imgY2 - txtY1);
    //写真との間にspaceができるように下に移動
    const offset = (origImgLength * capOffsetCoef) * userSettingObj.captionOffset;
    txtLyr.translate(0, offset);
    //縦写真は90度右回転して戻す
    if (origImgH>origImgW && userSettingObj.aspectRatio!=null) {
        docObj.rotateCanvas(90);
    }
    return txtLyr;
}

function resizeImage(docObj, userSettingObj) {
    //レイヤーを結合
    docObj.flatten();
    //白ワクのアスクペクト比が横長の場合
    if (docObj.width >= docObj.height) {
        //幅に合わせて倍率を決める
        resizePicture(docObj, userSettingObj.resize / docObj.width);
    //白ワクのアスクペクト比が縦長の場合
    } else if (docObj.width < docObj.height) {
        //高さに合わせて倍率を決める
        resizePicture(docObj, userSettingObj.resize / docObj.height);
    }
}

function resizePicture(docObj, scaleFactor) {
    docObj.resizeImage(docObj.width * scaleFactor, 
                        docObj.height * scaleFactor, 
                        docObj.resolution, 
                        ResampleMethod.BICUBIC);
}

function savePicture(docObj, userSettingObj) {
    //拡張子があれば除去してfilenameにする
    if (docObj.name.indexOf(".") == -1) {
        const docFileName = docObj.name;
    } else {
        docFileName = docObj.name.substring(0, docObj.name.lastIndexOf("."));
    }
    //保存ディレクトリ
    const savFilePath = docObj.path + "/frame";
    //あるか確認、なければ作る
    if (!(new Folder(savFilePath)).exists) {
        (new Folder(savFilePath)).create();
    }
    //保存ファイル名
    const savFileName = docFileName + "_frame";
    const savFile = new File(savFilePath + "/" + savFileName); //Fileをnew

    //保存用に画像を複製
    docObj.duplicate(docFileName + "_duplicated", false); //複製を実行
    const docObjDup =  app.activeDocument;

    //出力前の色空間
    //var colorProfileBefore = docDup.colorProfileName;

    //JPEGSaveOptionsの設定
    const jpgSavOpts = new JPEGSaveOptions(); //JPEG保存設定をnew
    jpgSavOpts.embedColorProfile = true;    //カラープロファイルの埋め込み
    jpgSavOpts.quality = 12;                //画質0～12
    jpgSavOpts.formatOptions = FormatOptions.PROGRESSIVE;
    //画像が読み込まれる時に上から順に表示されるのがベースライン、
    //全体が表示されるけど粗くてだんだん鮮明になっていくのがプログレッシブ
    jpgSavOpts.scans = 3;                   //スキャンの段階数 3～5
    jpgSavOpts.matte = MatteType.NONE;

    //画像保存
    activeDocument.saveAs(savFile, jpgSavOpts, true, Extension.LOWERCASE);
    //出力後の色空間
    //var colorProfileAfter = docDup.colorProfileName;
    //alert("出力前の色空間" + colorProfileBefore +"\n"+
    //      "出力後の色空間" + colorProfileBefore       );

    //保存用画像を閉じる
    docObjDup.close(SaveOptions.DONOTSAVECHANGES);
    docObjDup = null;
}
